<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

include "../../conn.php";
include "../../functions2.php";

header('Content-Type: application/json; charset=utf-8');
header('Strict-Transport-Security: max-age=31536000');
header('Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization');
header('Access-Control-Allow-Credentials: true');

/**
 * ✅ Safer CORS: whitelist (আপনার ডোমেইন বসাবেন)
 */
$allowedOrigins = [
    // "https://royelclub01.top",
    // "https://royelclub01.top",
];

$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
if (!empty($allowedOrigins) && in_array($origin, $allowedOrigins, true)) {
    header('Access-Control-Allow-Origin: ' . $origin);
} else {
    header('Access-Control-Allow-Origin: ' . ($origin ?: '*'));
}
header('Vary: Origin');

date_default_timezone_set("Asia/Dhaka");
$now = date("Y-m-d H:i:s");

/* ✅ UPDATE: wallet number middle 4 digit hide */
function mask_wallet_number($number) {
    $num = preg_replace('/\s+/', '', (string)$number);
    $len = strlen($num);

    if ($len <= 6) return $num;

    $start = 3; // first 3 digits visible
    $hide  = 4; // hide 4 digits
    $end   = $len - ($start + $hide);

    if ($end < 2) {
        $end = 2;
        $start = max(1, $len - ($hide + $end));
    }

    return substr($num, 0, $start)
        . str_repeat('*', $hide)
        . substr($num, $len - $end);
}

$response = [
    'code' => 11,
    'msg' => 'Method not allowed',
    'msgCode' => 12,
    'serviceNowTime' => $now,
];

/* ✅ Only POST allowed */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

$body = file_get_contents("php://input");
$post = json_decode($body, true);

/* Debug log */
file_put_contents(__DIR__ . '/withdrawals_debug.log',
    "POST: " . print_r($post, true) . "\n",
    FILE_APPEND
);

if (!is_array($post)) {
    http_response_code(400);
    echo json_encode([
        'code' => 7,
        'msg'  => 'Invalid JSON body'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// ===============================================
//  AUTO FIX: Create last_wager column if missing
// ===============================================
$checkCol = $conn->query("SHOW COLUMNS FROM shonu_kaichila LIKE 'last_wager'");
if ($checkCol && $checkCol->num_rows == 0) {
    $alter = "ALTER TABLE shonu_kaichila ADD COLUMN last_wager INT NOT NULL DEFAULT 0";
    if ($conn->query($alter)) {
        file_put_contents(__DIR__ . '/withdrawals_debug.log',
            "AUTO-FIX APPLIED: last_wager column created\n",
            FILE_APPEND
        );
    } else {
        file_put_contents(__DIR__ . '/withdrawals_debug.log',
            "AUTO-FIX FAILED: Could not add last_wager column\n",
            FILE_APPEND
        );
    }
}

/* ==========================
   REQUIRED PARAMETERS CHECK
   ========================== */
$required = ['language', 'random', 'signature', 'timestamp', 'withdrawid'];
foreach ($required as $r) {
    if (!isset($post[$r])) {
        http_response_code(400);
        echo json_encode([
            'code' => 7,
            'msg'  => 'Parameter invalid'
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
}

$languageRaw   = $post['language'];
$randomRaw     = $post['random'];
$signatureRaw  = $post['signature'];
$withdrawidRaw = $post['withdrawid'];

/* ✅ Cast / sanitize */
$language   = (string)$languageRaw;
$random     = (string)$randomRaw;
$signature  = (string)$signatureRaw;
$withdrawid = intval($withdrawidRaw);

/* ======================
   SIGNATURE VALIDATION
   ====================== */
$signString = '{"language":' . $language . ',"random":"' . $random . '","withdrawid":' . $withdrawid . '}';
$signCheck  = strtoupper(md5($signString));

file_put_contents(__DIR__ . '/withdrawals_debug.log',
    "SIGNATURE INPUT: $signString\nSIGNATURE CALC: $signCheck\nSIGNATURE RECV: $signature\n",
    FILE_APPEND
);

if (strtoupper($signature) !== $signCheck) {
    http_response_code(401);
    echo json_encode([
        'code' => 4,
        'msg'  => 'Invalid signature'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

/* ======================
   JWT AUTHENTICATION
   ====================== */
$bearer = explode(" ", $_SERVER['HTTP_AUTHORIZATION'] ?? "");
$jwt = $bearer[1] ?? "";

$jwtData = json_decode(is_jwt_valid($jwt), true);
if (!isset($jwtData['status']) || $jwtData['status'] !== 'Success') {
    http_response_code(401);
    echo json_encode([
        'code' => 4,
        'msg'  => 'Invalid Authorization'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

$userid = intval($jwtData['payload']['id'] ?? 0);
if ($userid <= 0) {
    http_response_code(401);
    echo json_encode([
        'code' => 4,
        'msg'  => 'Invalid user'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

$jwt_safe = mysqli_real_escape_string($conn, $jwt);

/* ✅ Check authorization in DB */
$auth = $conn->query("SELECT akshinak FROM shonu_subjects WHERE akshinak='$jwt_safe' LIMIT 1");
if (!$auth || $auth->num_rows != 1) {
    http_response_code(401);
    echo json_encode([
        'code' => 4,
        'msg'  => 'No operation permission'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

/* ======================
      WITHDRAW BANK LIST
   ====================== */
$data = [];
$data['withdrawalslist'] = [];

/**
 * ✅ UPDATE: সব withdrawid-এ wallet দেখাতে চাইলে নিচের if বাদ দিন
 * (আপনার আগের সমস্যা: wallet show হচ্ছিল না)
 */
$SHOW_WALLET_ALWAYS = true;

if ($SHOW_WALLET_ALWAYS || $withdrawid === 4) {
    $bankJson = '[
        {"bankID":175,"bankLogo":"https://spg.xworldpp.com/banks/bKash.png","bankName":"BKASH"},
        {"bankID":178,"bankLogo":"https://spg.xworldpp.com/banks/Nagad.png","bankName":"NAGAD"},
        {"bankID":176,"bankLogo":"https://upload.wikimedia.org/wikipedia/commons/e/e9/Rocket_ddbl.png","bankName":"ROCKET"},
        {"bankID":177,"bankLogo":"https://upload.wikimedia.org/wikipedia/commons/d/d5/Upay_logo.svg","bankName":"UPAY"},
        {"bankID":237,"bankLogo":"https://ossimg.dkwinpicture.com/dkwin","bankName":"Surecash"}
    ]';

    $bankList = json_decode($bankJson, true);
    $bankMap = [];
    foreach ($bankList as $b) $bankMap[intval($b['bankID'])] = $b['bankName'];

    $q = $conn->query("SELECT * FROM bankcard WHERE userid=$userid");
    if ($q) {
        while ($row = $q->fetch_assoc()) {
            $type = intval($row['type']);
            $name = $bankMap[$type] ?? "E-Wallet";

            $data['withdrawalslist'][] = [
                "bid" => intval($row['id']),
                "beneficiaryName" => $row['name'],

                // ✅ UPDATE HERE: mask 4 digits
                "mobileNO" => mask_wallet_number($row['account']),

                "walletName" => $name,
                "withType" => $type
            ];
        }
    }
}

/* ======================
       WITHDRAW COUNT
   ====================== */
$todayCountRes = $conn->query("
    SELECT shonu FROM hintegedukolli
    WHERE balakedara=$userid
    AND DATE(dinankavannuracisi)=CURDATE()
");
$todayCount = $todayCountRes ? $todayCountRes->num_rows : 0;

$remaining = ($withdrawid === 3) ? max(0, 5 - $todayCount) :
             (($withdrawid === 1) ? max(0, 3 - $todayCount) : 0);

/* ======================
       USER BALANCE
   ====================== */
$balRes = $conn->query("SELECT * FROM shonu_kaichila WHERE balakedara=$userid LIMIT 1");
$bal = $balRes ? $balRes->fetch_assoc() : [];

$balance = floatval($bal['motta'] ?? 0);
$bonus   = floatval($bal['bonus'] ?? 0);
$rebet   = floatval($bal['rebet'] ?? 0);
$spin    = floatval($bal['spin'] ?? 0);
$claimed = intval($bal['bonus_claimed'] ?? 0);
$prevLastWager = intval($bal['last_wager'] ?? 0);

/* ======================
        TOTAL DEPOSIT
   ====================== */
$dep = $conn->query("SELECT SUM(motta) AS t FROM thevani WHERE balakedara=$userid AND sthiti='1'");
$depRow = $dep ? $dep->fetch_assoc() : [];
$totalDeposit = floatval($depRow['t'] ?? 0);

/* ======================
  LATEST APPROVED DEPOSIT
   ====================== */
$latestRes = $conn->query("
    SELECT motta, dinankavannuracisi
    FROM thevani
    WHERE balakedara=$userid AND sthiti='1'
    ORDER BY shonu DESC LIMIT 1
");
$latest = $latestRes ? $latestRes->fetch_assoc() : [];

$latestAmount = floatval($latest['motta'] ?? 0);
$latestTime   = $latest['dinankavannuracisi'] ?? null;

/* ======================
   ✅ CLEAN + FINAL WAGER BLOCK
   ====================== */

// ✅ All wager tables
$tables = [
    'bajikattuttate',
    'bajikattuttate_drei',
    'bajikattuttate_funf',
    'bajikattuttate_zehn',
    'bajikattuttate_trx',
    'bajikattuttate_trx3',
    'bajikattuttate_trx5',
    'bajikattuttate_trx10',
    'bajikattuttate_kemuru',
    'bajikattuttate_kemuru_drei',
    'bajikattuttate_kemuru_funf',
    'bajikattuttate_kemuru_zehn',
    'bajikattuttate_aidudi',
    'bajikattuttate_aidudi_drei',
    'bajikattuttate_aidudi_funf',
    'bajikattuttate_aidudi_zehn',
];

/**
 * ✅ IMPORTANT:
 * Deposit-time filter অনেক গেমে mismatch হওয়ায় wager count হচ্ছিল না।
 * তাই default false (সব গেম wager এ count হবে)
 * যদি আপনার দরকার হয়, true করে দিন।
 */
$USE_DEPOSIT_TIME_FILTER = false;

$totalBet = 0;

foreach ($tables as $t) {

    $whereTime = "";
    if ($USE_DEPOSIT_TIME_FILTER && !empty($latestTime)) {
        $safeTime = mysqli_real_escape_string($conn, $latestTime);
        $whereTime = " AND tiarikala >= '$safeTime' ";
    }

    $sql = "SELECT COALESCE(SUM(ketebida),0) AS total
            FROM $t
            WHERE byabaharkarta=$userid $whereTime";

    $q = $conn->query($sql);

    // ✅ query fail হলে log
    if (!$q) {
        file_put_contents(__DIR__ . '/withdrawals_debug.log',
            "BET QUERY FAILED: $t | SQL: $sql | ERROR: " . $conn->error . "\n",
            FILE_APPEND
        );
        continue;
    }

    $row = $q->fetch_assoc();
    $bet = floatval($row['total'] ?? 0);

    // ✅ multiplier normalize (strpos fixed)
    $is3  = (strpos($t, 'trx3') !== false)  || (strpos($t, '_drei') !== false);
    $is5  = (strpos($t, 'trx5') !== false)  || (strpos($t, '_funf') !== false);
    $is10 = (strpos($t, 'trx10') !== false) || (strpos($t, '_zehn') !== false);
    $is20 = (strpos($t, '_aidudi') !== false) || (strpos($t, '_kemuru') !== false);

    $mult = $is3 ? 3 : ($is5 ? 5 : ($is10 ? 10 : ($is20 ? 20 : 1)));

    // normalize to base units
    $totalBet += ($mult > 1) ? ($bet / $mult) : $bet;
}

/* Required wager amount (আগের logic 그대로) */
$wagerBonus = ($bonus > 0)
    ? intval(($latestAmount + $bonus) * 20)
    : intval($latestAmount);

/* wager remaining */
$base = ($bonus == 0)
    ? intval($latestAmount)
    : intval(($latestAmount + $bonus) * 20);

$lastWager = max(0, $base - $totalBet);

// আগের rules 그대로
if ($balance < 5) $lastWager = 0;
if ($prevLastWager == 0 && $totalBet > 0) $lastWager = 0;

// update last_wager
$conn->query("UPDATE shonu_kaichila SET last_wager=" . intval($lastWager) . " WHERE balakedara=$userid");

// eligible
$eligible = ($totalBet >= $wagerBonus) ? $balance : 0;

// limits
$maxLimit = ($claimed === 1) ? 6000 : 30000;

// ✅ wager complete না হওয়া পর্যন্ত canWithdrawAmount = 0
$canWithdrawAmount = min($eligible, $maxLimit);

// optional debug
file_put_contents(__DIR__ . '/withdrawals_debug.log',
    "WAGER DEBUG | user=$userid | totalBet=$totalBet | required=$wagerBonus | lastWager=$lastWager | eligible=$eligible | canWithdrawAmount=$canWithdrawAmount\n",
    FILE_APPEND
);

/* ======================
      FINAL RESPONSE
   ====================== */
$data['withdrawalsrule'] = [
    "withdrawCount" => $todayCount,
    "withdrawRemainingCount" => $remaining,
    "startTime" => "00:00",
    "endTime" => "23:59",
    "minPrice" => 500,
    "maxPrice" => $maxLimit,
    "canWithdrawAmount" => $canWithdrawAmount,
    "canWithdrawEligible" => $eligible,
    "requiredWagerBonus" => $wagerBonus,
    "amountofCode" => intval($lastWager),
    "wagerProgress" => $totalBet . " / " . $wagerBonus,
    "wagerProgressPercent" => ($wagerBonus > 0 ? round(($totalBet / $wagerBonus) * 100, 2) . "%" : "0%"),
    "totalDeposit" => $totalDeposit,
    "firstBonus" => $bonus,
    "amount" => $balance,
    "motta" => $balance,
    "balance" => $balance,
    "uRate" => 93,
    "uGold" => 0
];

$data['debug'] = [
    'total_deposit'   => $totalDeposit,
    'first_bonus'     => $bonus,
    'wager_bonus'     => $wagerBonus,
    'total_bet'       => $totalBet,
    'last_wager_base' => $base,
    'user_balance'    => $balance,
    'latest_deposit'  => $latestAmount,
    'latest_time'     => $latestTime
];

$response['data'] = $data;
$response['balance'] = $balance;
$response['wallet']  = $balance;
$response['code'] = 0;
$response['msg'] = ($eligible > 0)
    ? "আপনার ওয়েজার সম্পূর্ণ হয়েছে, এখন উত্তোলন করতে পারবেন।"
    : "ওয়েজার বাকি আছে, উত্তোলনের আগে সম্পূর্ণ করতে হবে।";
$response['msgCode'] = 0;

http_response_code(200);
echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
exit;
?>
